# Info:
# Script requires R="4.3.1" and ggplot2="3.4.3" to function properly (you can use for example r-lib rig to change your default R version: https://github.com/r-lib/rig)
# Replace "path/to/.../" with the according path to the input-file or output directory
# feed supplementary tables 1 and 2 to the script, which can be found here: https://osf.io/n5qj6/

# On Ubuntu you need to install following library first: libgdal-dev & libfontconfig1-dev
# sudo apt install libgdal-dev
# sudo apt install libfontconfig1-dev

# Install "remotes"-package for versionized installation of necessary packages
install.packages("remotes")

# Install necessary packages with specific versions
remotes::install_version("ggplot2", version = "3.4.3")
remotes::install_version("ggnewscale", version = "0.4.10")
remotes::install_version("maptools", version = "1.1-8")
remotes::install_version("rgeos", version = "0.6-4")

# Install necessary packages
install.packages("dplyr")
install.packages("cowplot")
install.packages("ggmap")
install.packages("ggpolypath")
install.packages("gridExtra")
install.packages("raster")
install.packages("scales")
install.packages("stringr")
install.packages("svglite")

# Load libraries
library(ggplot2)
library(cowplot)
library(dplyr)
library(ggmap)
library(ggnewscale)
library(ggpolypath)
library(gridExtra)
library(maptools)
library(rgeos)
library(raster)
library(scales)
library(stringr)

########################################################################################
# Fetch data for federal states
# If raster::getData is not working you can also load "geoData.RData" from https://osf.io/n5qj6/
FEDERAL_STATES.shp <- raster::getData("GADM", country = "DEU", level = 1)
FEDERAL_STATES_DF <- fortify(FEDERAL_STATES.shp)

# Fetch data for Landkreise
DISTRICTS.shp <- raster::getData("GADM", country = "DEU", level = 2)
# Transform fetched Landkreis-datainto dataframe & filter for all Thuringian-Landkreise
DISTRICTS_DF <- fortify(DISTRICTS.shp, region = "CC_2")
DISTRICTS_DF_THURINGIA <- DISTRICTS_DF[DISTRICTS_DF$id >= "16000",] # Only Thuringian Landkreise carry an Id > 16000

# create & modify df
ALPHA_AMOUNT_PER_MONTH_FEDERAL_STATE_DF <- read.csv(file = "/path/to/supplementary_table_1-german_alpha-amount_per_month_federal-state.tsv", sep = '\t')
ALPHA_AMOUNT_PER_MONTH_FEDERAL_STATE_DF$id <- as.factor(ALPHA_AMOUNT_PER_MONTH_FEDERAL_STATE_DF[["id"]])
ALPHA_AMOUNT_PER_MONTH_FEDERAL_STATE_TIME_FILTERED_DF <- filter(ALPHA_AMOUNT_PER_MONTH_FEDERAL_STATE_DF, Month %in% c("2020-Dec-2nd-half", "2021-Jan-1st-half", "2021-Jan-2nd-half", "2021-Feb", "2021-Mar", "2021-Apr", "2021-May", "2021-Jun", "2021-Jul", "2021-Aug"))

THURINGIA_AMOUNT_PER_MONTH_PER_DISTRICT_DF <- read.csv(file = "/path/to/supplementary_table_2-thuringian_alpha-amount_per_month_per_district.tsv", sep = '\t')
THURINGIA_AMOUNT_PER_MONTH_PER_DISTRICT_DF$id <- as.factor(THURINGIA_AMOUNT_PER_MONTH_PER_DISTRICT_DF[["id"]])
THURINGIA_AMOUNT_PER_MONTH_PER_DISTRICT_TIME_FILTERED_DF <- filter(THURINGIA_AMOUNT_PER_MONTH_PER_DISTRICT_DF, Month %in% c("2020-Dec-2nd-half", "2021-Jan-1st-half", "2021-Jan-2nd-half", "2021-Feb", "2021-Mar", "2021-Apr", "2021-May", "2021-Jun", "2021-Jul", "2021-Aug"))

# merge df with FEDERAL_STATE.shp
MERGED_DF_GERMANY_FEDERAL_STATES <- full_join(FEDERAL_STATES_DF, ALPHA_AMOUNT_PER_MONTH_FEDERAL_STATE_TIME_FILTERED_DF, by = "id")
MERGED_DF_GERMANY_FEDERAL_STATES <- na.omit(MERGED_DF_GERMANY_FEDERAL_STATES)
MERGED_DF_GERMANY_FEDERAL_STATES["percentage_Alpha"][MERGED_DF_GERMANY_FEDERAL_STATES["n_total"] == 0] <- NA
MERGED_DF_GERMANY_FEDERAL_STATES["n_total"][MERGED_DF_GERMANY_FEDERAL_STATES["n_total"] == 0] <- NA
MERGED_DF_GERMANY_FEDERAL_STATES["Month"][MERGED_DF_GERMANY_FEDERAL_STATES["Month"] == "2020-Dec-1st-half"] <- "2020-Dec\n1st-half"
MERGED_DF_GERMANY_FEDERAL_STATES["Month"][MERGED_DF_GERMANY_FEDERAL_STATES["Month"] == "2020-Dec-2nd-half"] <- "2020-Dec\n2nd-half"
MERGED_DF_GERMANY_FEDERAL_STATES["Month"][MERGED_DF_GERMANY_FEDERAL_STATES["Month"] == "2021-Jan-1st-half"] <- "2021-Jan\n1st-half"
MERGED_DF_GERMANY_FEDERAL_STATES["Month"][MERGED_DF_GERMANY_FEDERAL_STATES["Month"] == "2021-Jan-2nd-half"] <- "2021-Jan\n2nd-half"
MERGED_DF_GERMANY_FEDERAL_STATES$month_factor <- factor(MERGED_DF_GERMANY_FEDERAL_STATES$Month, levels = c("2020-Dec\n2nd-half", "2021-Jan\n1st-half", "2021-Jan\n2nd-half", "2021-Feb", "2021-Mar", "2021-Apr", "2021-May", "2021-Jun", "2021-Jul", "2021-Aug"), ordered = TRUE)

MERGED_DF_THURINGIA_DISTRICTS <- full_join(DISTRICTS_DF_THURINGIA, THURINGIA_AMOUNT_PER_MONTH_PER_DISTRICT_TIME_FILTERED_DF, by = "id")
MERGED_DF_THURINGIA_DISTRICTS <- na.omit(MERGED_DF_THURINGIA_DISTRICTS)
MERGED_DF_THURINGIA_DISTRICTS["percentage_Alpha"][MERGED_DF_THURINGIA_DISTRICTS["n_total"] == 0] <- NA
MERGED_DF_THURINGIA_DISTRICTS["n_total"][MERGED_DF_THURINGIA_DISTRICTS["n_total"] == 0] <- NA
MERGED_DF_THURINGIA_DISTRICTS["Month"][MERGED_DF_THURINGIA_DISTRICTS["Month"] == "2020-Dec-1st-half"] <- "2020-Dec\n1st-half"
MERGED_DF_THURINGIA_DISTRICTS["Month"][MERGED_DF_THURINGIA_DISTRICTS["Month"] == "2020-Dec-2nd-half"] <- "2020-Dec\n2nd-half"
MERGED_DF_THURINGIA_DISTRICTS["Month"][MERGED_DF_THURINGIA_DISTRICTS["Month"] == "2021-Jan-1st-half"] <- "2021-Jan\n1st-half"
MERGED_DF_THURINGIA_DISTRICTS["Month"][MERGED_DF_THURINGIA_DISTRICTS["Month"] == "2021-Jan-2nd-half"] <- "2021-Jan\n2nd-half"
MERGED_DF_THURINGIA_DISTRICTS$month_factor <- factor(MERGED_DF_THURINGIA_DISTRICTS$Month, levels = c("2020-Dec\n2nd-half", "2021-Jan\n1st-half", "2021-Jan\n2nd-half", "2021-Feb", "2021-Mar", "2021-Apr", "2021-May", "2021-Jun", "2021-Jul", "2021-Aug"), ordered = TRUE)

########################################################################################
# Joined heatmaps for total sample count & Alpha-lineage proportion
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP <- ggplot() +
  geom_polypath(data = MERGED_DF_GERMANY_FEDERAL_STATES, aes(x = long, y = lat, group = group, fill = n_total, colour = ""), size = 0.15) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#ede8ed", "#dbd1da", "#b19cb0", "#785171", "#4a3344", "#2e262c"),
    values = c(0, 0.2, 0.4, 0.6, 0.8, 1),
    trans = scales::pseudo_log_trans(sigma = 1),
    breaks = c(NA, 10, 100, 1000, 10000, NA),
    labels= abs, limits = c(1, 100000),
    guide = guide_colourbar(
      title = "Sequenced SARS-CoV-2 isolates",
      title.position = "top",
      direction = "horizontal",
      title.vjust = 1.5,
      title.hjust = 2,
      title.theme = element_text(size = 11),
      label.vjust = -2,
      label.theme = element_text(size = 10),
      barwidth = unit(5, "cm"),
      order = 2
    )
  ) +
  scale_colour_manual(values = "#ffffff", labels = "NA") +
  guides(colour = guide_legend("", title.position = "top", label.vjust = 3, order = 2, override.aes = list(fill = "#adadad"))) +
  facet_grid(rows = vars(month_factor)) +
  theme(
    legend.position = "bottom",
    legend.box="vertical",
    legend.title = element_text(size = 10),
    legend.text = element_text(size = 7),
    legend.key.size = unit(0.3, "cm"),
    legend.spacing.y = unit(-0.1, "cm"),
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1.2,
    plot.margin = unit(c(0.1,-2,0.1,-2.8), "cm"),
    strip.text.y = element_blank()
  )

GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP

ISOLATE_COUNT_LEGEND <- get_legend(GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP)
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP <- GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP +
  theme(legend.position = "none")
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP

THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP <- ggplot() +
  geom_polypath(data = MERGED_DF_THURINGIA_DISTRICTS, aes(x = long, y = lat, group = group, fill = n_total, colour = ""), size = 0.15) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#ede8ed", "#dbd1da", "#b19cb0", "#785171", "#4a3344", "#2e262c"),
    values = c(0, 0.2, 0.4, 0.6, 0.8, 1),
    trans = scales::pseudo_log_trans(sigma = 1),
    breaks = c(1, 10, 100, 1000),
    limits = c(1, 100000)
  ) +  scale_colour_manual(values = "#ffffff", labels = "NA") +
  guides(colour = guide_legend("", title.position = "top", override.aes = list(fill = "#adadad", label.vjust = 3, order = 2))) +
  facet_grid(rows = vars(month_factor)) +
  theme(
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1,
    plot.margin = unit(c(0.1,-1.5,0.1,-2.7), "cm"),
    strip.text.y = element_blank()
  )

THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP

GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP <- ggplot() +
  geom_polypath(data = MERGED_DF_GERMANY_FEDERAL_STATES, aes(x = long, y = lat, group = group, fill = percentage_Alpha, colour = ""), size = 0.15) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#edebbb", "#f0c880", "#f09d58", "#dd745e", "#c3566b", "#943e6c"),
    values = c(0, 0.5, 1),
    breaks = c(NA, 25, 50, 75, NA),
    limits = c(0, 100),
    guide = guide_colourbar(
      title = "Proportion of Alpha lineage",
      title.position = "top",
      direction = "horizontal",
      title.vjust = 1.5,
      title.hjust = 0.5,
      title.theme = element_text(size = 11),
      label.vjust = -0.5,
      label.theme = element_text(size = 10),
      barwidth = unit(3.8, "cm"),
      order = 2
    )
  ) +
  scale_colour_manual(values = "#ffffff", labels = "NA") +
  guides(
    colour = guide_legend(
      " ",
      title.position = "top",
      title.theme = element_text(size = 10),
      label.position = "bottom",
      label.vjust = -0.5,
      label.theme = element_text(size = 9),
      order = 1,
      override.aes = list(fill = "#adadad")
    )
  ) +
  facet_grid(rows = vars(month_factor)) +
  theme(
    legend.position = "bottom",
    legend.box="horizontal",
    legend.key.size = unit(0.3, "cm"),
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1.2,
    plot.margin = unit(c(0.1,-2,0.1,-3.0), "cm"),
    strip.text.y = element_blank()
  )

GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP

ALPHA_PROPORTION_LEGEND <- get_legend(GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP)
GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP <- GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP +
  theme(legend.position = "none")
GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP

THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP <- ggplot() +
  geom_polypath(data = MERGED_DF_THURINGIA_DISTRICTS, aes(x = long, y = lat, group = group, fill = percentage_Alpha, colour = ""), size = 0.15) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#edebbb", "#f0c880", "#f09d58", "#dd745e", "#c3566b", "#943e6c"),
    values = c(0, 0.5, 1),
    breaks = c(0, 25, 50, 75, 100),
    limits = c(0, 100)
  ) +
  scale_colour_manual(values = "#ffffff", labels = "NA") +
  guides(colour = guide_legend("", title.position = "top", override.aes = list(fill = "#adadad", label.vjust = 3, order = 2))) +
  facet_grid(rows = vars(month_factor)) +
  theme(
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1,
    plot.margin = unit(c(0.1,-1,0.1,-1.6), "cm"),
    strip.background = element_blank()
  )
THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP

GERMAN_FEDERAL_STATES_HEATMAPS <- grid.arrange(GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP, GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP, ncol = 2)
THURINGIAN_DISTRICTS_HEATMAPS <- grid.arrange(THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP, THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP, ncol = 2)
COMPLETE_HEATMAPS <- grid.arrange(GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP, GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP, THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP, THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP, ncol = 4)
COMPLETE_HEATMAPS

#ggsave("germany_joined_heatmap.svg", plot = GERMAN_FEDERAL_STATES_HEATMAPS, path = "/path/to/output", width = 1000, height = 3000, units = "px", dpi = 300, scale = 1.3)
#ggsave("thuringia_joined_heatmap.svg", plot = THURINGIAN_DISTRICTS_HEATMAPS, path = "/path/to/output", width = 1000, height = 3000, units = "px", dpi = 300, scale = 1.3)
ggsave("complete_joined_heatmap.svg", plot = COMPLETE_HEATMAPS, path = "/path/to/output", width = 1200, height = 3000, units = "px", dpi = 300, scale = 1.3)
ggsave("legend_isolate_count.svg", plot = ISOLATE_COUNT_LEGEND, path = "/path/to/output", width = 800, height = 200, units = "px", dpi = 300, scale = 1.3)
ggsave("legend_alpha_proportion.svg", plot = ALPHA_PROPORTION_LEGEND, path = "/path/to/output", width = 600, height = 200, units = "px", dpi = 300, scale = 1.3)

########################################################################################
# Joined heatmaps for total sample count & Alpha-lineage proportion - Dec'20 till Mar'21
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART <- ggplot() +
  geom_polypath(
    data = filter(MERGED_DF_GERMANY_FEDERAL_STATES, Month %in% c("2020-Dec\n2nd-half", "2021-Jan\n1st-half", "2021-Jan\n2nd-half", "2021-Feb", "2021-Mar")),
    aes(x = long, y = lat, group = group, fill = n_total, colour = ""),
    size = 0.15
  ) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#ede8ed", "#dbd1da", "#b19cb0", "#785171", "#4a3344", "#2e262c"),
    values = c(0, 0.2, 0.4, 0.6, 0.8, 1),
    trans = scales::pseudo_log_trans(sigma = 1),
    breaks = c(NA, 10, 100, 1000, 10000, NA),
    labels= abs, limits = c(1, 100000),
    guide = guide_colourbar(
      title = "Sequenced SARS-CoV-2 isolates",
      title.position = "top",
      direction = "horizontal",
      title.vjust = 1.5,
      title.hjust = 2,
      title.theme = element_text(size = 11),
      label.vjust = -2,
      label.theme = element_text(size = 10),
      barwidth = unit(5, "cm"),
      order = 2
    )
  ) +
  scale_colour_manual(
    values = "#ffffff",
    labels = "NA"
  ) +
  guides(
    colour = guide_legend(
      " ",
      title.position = "top",
      title.theme = element_text(size = 10),
      label.position = "bottom",
      label.vjust = -1,
      label.theme = element_text(size = 9),
      order = 1,
      override.aes = list(fill = "#adadad")
    )
  ) +
  facet_grid(rows = vars(month_factor), labeller = label_wrap_gen(12)) +
  theme(
    legend.position = "bottom",
    legend.box="horizontal",
    legend.key.size = unit(0.3, "cm"),
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1.2,
    plot.margin = unit(c(0.1,-2,0.1,-2.8), "cm"),
    strip.text.y = element_blank()
  )
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART

ISOLATE_COUNT_LEGEND <- get_legend(GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART)
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART <- GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART +
  theme(legend.position = "none")
GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART

THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP_PART <- ggplot() +
  geom_polypath(
    data = filter(MERGED_DF_THURINGIA_DISTRICTS, Month %in% c("2020-Dec\n2nd-half", "2021-Jan\n1st-half", "2021-Jan\n2nd-half", "2021-Feb", "2021-Mar")),
    aes(x = long, y = lat, group = group, fill = n_total, colour = ""),
    size = 0.15) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#ede8ed", "#dbd1da", "#b19cb0", "#785171", "#4a3344", "#2e262c"),
    values = c(0, 0.2, 0.4, 0.6, 0.8, 1),
    trans = scales::pseudo_log_trans(sigma = 1),
    breaks = c(1, 10, 100, 1000),
    limits = c(1, 100000)
  ) +
  scale_colour_manual(values = "#ffffff", labels = "NA") +
  facet_grid(rows = vars(month_factor), labeller = label_wrap_gen(12)) +
  theme(
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1,
    plot.margin = unit(c(0.1,-1.5,0.1,-2.7), "cm"),
    strip.text.y = element_blank()
  )
THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP_PART

GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART <- ggplot() +
  geom_polypath(
    data = filter(MERGED_DF_GERMANY_FEDERAL_STATES, Month %in% c("2020-Dec\n2nd-half", "2021-Jan\n1st-half", "2021-Jan\n2nd-half", "2021-Feb", "2021-Mar")),
    aes(x = long, y = lat, group = group, fill = percentage_Alpha, colour = ""),
    size = 0.15
  ) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#edebbb", "#f0c880", "#f09d58", "#dd745e", "#c3566b", "#943e6c"),
    values = c(0, 0.5, 1),
    breaks = c(NA, 25, 50, 75, NA),
    limits = c(0, 100),
    guide = guide_colourbar(
      title = "Proportion of Alpha lineage",
      title.position = "top",
      direction = "horizontal",
      title.vjust = 1.5,
      title.hjust = 0.5,
      title.theme = element_text(size = 11),
      label.vjust = -0.5,
      label.theme = element_text(size = 10),
      barwidth = unit(3.8, "cm"),
      order = 2
    )
  ) +
  scale_colour_manual(values = "#ffffff", labels = "NA") +
  guides(
    colour = guide_legend(
      " ",
      title.position = "top",
      title.theme = element_text(size = 10),
      label.position = "bottom",
      label.vjust = -0.5,
      label.theme = element_text(size = 9),
      order = 1,
      override.aes = list(fill = "#adadad")
    )
  ) +
  facet_grid(rows = vars(month_factor), labeller = label_wrap_gen(12)) +
  theme(
    legend.position = "bottom",
    legend.box="horizontal",
    legend.key.size = unit(0.3, "cm"),
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1.2,
    plot.margin = unit(c(0.1,-2,0.1,-3.0), "cm"),
    strip.text.y = element_blank()
  )
GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART

ALPHA_PROPORTION_LEGEND <- get_legend(GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART)
GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART <- GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART +
  theme(legend.position = "none")
GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART

THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP_PART <- ggplot() +
  geom_polypath(
    data = filter(MERGED_DF_THURINGIA_DISTRICTS, Month %in% c("2020-Dec\n2nd-half", "2021-Jan\n1st-half", "2021-Jan\n2nd-half", "2021-Feb", "2021-Mar")),
    aes(x = long, y = lat, group = group, fill = percentage_Alpha, colour = "NA"),
    size = 0.15
  ) +
  scale_fill_gradientn(
    na.value = "#adadad",
    colours = c("#edebbb", "#f0c880", "#f09d58", "#dd745e", "#c3566b", "#943e6c"),
    values = c(0, 0.5, 1),
    breaks = c(0, 25, 50, 75, 100),
    limits = c(0, 100)
  ) +
  scale_colour_manual(values = "#ffffff", labels = "NA") +
  facet_grid(rows = vars(month_factor), labeller = label_wrap_gen(12)) +
  theme(
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1,
    plot.margin = unit(c(0.1,-1,0.1,-1.6), "cm"),
    strip.background = element_blank()
  )
THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP_PART

GERMAN_FEDERAL_STATES_HEATMAPS_PART <- grid.arrange(GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART, GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART, ncol = 2)
THURINGIAN_DISTRICTS_HEATMAPS_PART <- grid.arrange(THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP_PART, THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP_PART, ncol = 2)
COMPLETE_HEATMAPS_PART <- grid.arrange(GERMAN_FEDERAL_STATES_TOTAL_ISOLATES_HEATMAP_PART, GERMAN_FEDERAL_STATES_ALPHA_PERCENTAGE_HEATMAP_PART, THURINGIAN_DISTRICTS_TOTAL_ISOLATES_HEATMAP_PART, THURINGIAN_DISTRICTS_ALPHA_PERCENTAGE_HEATMAP_PART, ncol = 4)
COMPLETE_HEATMAPS_PART

#ggsave("germany_joined_heatmap_partial.png", plot = GERMAN_FEDERAL_STATES_HEATMAPS_PART, path = "/path/to/output", width = 1000, height = 1500, units = "px", dpi = 300, scale = 1.3)
#ggsave("thuringia_joined_heatmap_partial.png", plot = THURINGIAN_DISTRICTS_HEATMAPS_PART, path = "/path/to/output", width = 1000, height = 1500, units = "px", dpi = 300, scale = 1.3)
ggsave("complete_joined_heatmap_partial_no_legend.svg", plot = COMPLETE_HEATMAPS_PART, path = "/path/to/output", width = 1150, height = 1500, units = "px", dpi = 300, scale = 1.3)
ggsave("legend_isolate_count.svg", plot = ISOLATE_COUNT_LEGEND, path = "/path/to/output", width = 800, height = 200, units = "px", dpi = 300, scale = 1.3)
ggsave("legend_alpha_proportion.svg", plot = ALPHA_PROPORTION_LEGEND, path = "/path/to/output", width = 600, height = 200, units = "px", dpi = 300, scale = 1.3)


########################################################################################
# Map of Germany with Thuringia highlighted

THURINGIAN_MAP <- ggplot() +
  geom_polypath(data = FEDERAL_STATES_DF, aes(x = long, y = lat, group = group), fill = "#849094", colour = "black", size = 0.3) +
  geom_polypath(data = DISTRICTS_DF_THURINGIA, aes(x = long, y = lat, group = group), fill = "#943e6c", size = 0.4) +
  theme(
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1.2,
    strip.background = element_blank()
  )
THURINGIAN_MAP